package tech.atani.connections.handler;

import com.google.gson.JsonElement;
import com.google.gson.JsonParser;
import com.google.gson.JsonSyntaxException;
import tech.atani.connections.packets.Packet;
import tech.atani.connections.packets.client.C7HandshakeRequest;
import tech.atani.connections.packets.server.S7HandshakeResponse;
import tech.atani.entities.HandshakeUser;
import tech.atani.main.Main;
import tech.atani.server.Server;
import tech.atani.utils.AESUtil;

import java.io.IOException;
import java.io.PrintStream;
import java.net.Socket;
import java.security.Key;
import java.util.Base64;
import java.util.Scanner;

public class ConnectionHandler {

    private final Server server;
    private Thread handlerThread;

    public ConnectionHandler(Server server) {
        this.server = server;
    }

    public void start() {
        handlerThread = new Thread(() -> {
            while (!server.getServerSocket().isClosed()) {
                Main.getInstance().getLogger().debug("waiting");

                Socket clientSocket = null;

                try {
                    clientSocket = server.getServerSocket().accept();
                    Main.getInstance().getLogger().info("accepted an incoming connection");
                } catch (Exception ex) {
                    Main.getInstance().getLogger().info("failed accepting incoming connection");
                }

                if (clientSocket == null)
                    continue;

                Scanner inputScanner = null;

                try {
                    inputScanner = new Scanner(clientSocket.getInputStream());
                    Main.getInstance().getLogger().debug("created input scanner");
                } catch (IOException e) {
                    Main.getInstance().getLogger().info("failed creating input scanner");
                }

                if (inputScanner == null)
                    continue;

                if (!inputScanner.hasNextLine()) {
                    Main.getInstance().getLogger().debug("no line");

                    try {
                        inputScanner.close();
                        clientSocket.close();
                    } catch (IOException ignore) { }
                }

                String content = null;

                try {
                    content = inputScanner.nextLine();
                    Main.getInstance().getLogger().debug("getting content");

                    if (content == null) {
                        Main.getInstance().getLogger().info("invalid content: (" + clientSocket.getInetAddress().getHostAddress() + ")");

                        try {
                            inputScanner.close();
                            clientSocket.close();
                        } catch (Exception ignore) {
                        }
                        continue;
                    }

                    JsonElement jsonElement = JsonParser.parseString(content);

                    if (jsonElement.isJsonObject() && jsonElement.getAsJsonObject().has("id") && jsonElement.getAsJsonObject().has("content")) {
                        Main.getInstance().getLogger().debug("valid JSON.\n" + content);
                        Packet packet = server.getPacketManager().getPacket(jsonElement.getAsJsonObject(), null);
                        Main.getInstance().getLogger().debug("tried to resolve Packet.");

                        if (packet instanceof C7HandshakeRequest c7HandshakeRequest) {
                            Main.getInstance().getLogger().debug("received valid Handshake Packet.");

                            if (c7HandshakeRequest.isValid()) {
                                Main.getInstance().getLogger().debug("received valid format.");
                                Key aesKey;

                                try {
                                    aesKey = AESUtil.generateSecureKey();

                                    HandshakeUser handshakeUser = new HandshakeUser(clientSocket, aesKey, inputScanner);

                                    S7HandshakeResponse s7HandshakeResponse = new S7HandshakeResponse(Base64.getEncoder().encodeToString(aesKey.getEncoded()), true, "");
                                    server.getPacketManager().sendPacket(server, handshakeUser.getPrintStream(), s7HandshakeResponse);
                                    Main.getInstance().getLogger().info("user Handshake successful! (" + clientSocket.getInetAddress().getHostAddress() + ")");

                                    new LoginHandler(handshakeUser, server).startHandler();
                                } catch (Exception exception) {
                                    S7HandshakeResponse s7HandshakeResponse = new S7HandshakeResponse(null, false, exception.getMessage());

                                    server.getPacketManager().sendPacket(server, new PrintStream(clientSocket.getOutputStream()), s7HandshakeResponse);
                                    Main.getInstance().getLogger().info("user Handshake failed! (" + clientSocket.getInetAddress().getHostAddress() + ")");
                                }
                            } else {
                                Main.getInstance().getLogger().info("user Handshake failed! (" + clientSocket.getInetAddress().getHostAddress() + ", INVALID FORMAT)");

                                try {
                                    server.getPacketManager().sendPacket(server, new PrintStream(clientSocket.getOutputStream()), new S7HandshakeResponse(null, false, "Invalid Format."));
                                } catch (Exception ignore) {
                                }

                                try {
                                    inputScanner.close();
                                    clientSocket.close();
                                } catch (Exception ignore) { }
                            }
                        } else {
                            Main.getInstance().getLogger().info("received wrong Packet on connection (" + clientSocket.getInetAddress().getHostAddress() + ")");

                            try {
                                server.getPacketManager().sendPacket(server, new PrintStream(clientSocket.getOutputStream()), new S7HandshakeResponse(null, false, "Wrong Packet."));
                            } catch (Exception ignore) {
                            }

                            try {
                                inputScanner.close();
                                clientSocket.close();
                            } catch (IOException ignore) {
                            }
                        }
                    } else {
                        Main.getInstance().getLogger().info("Received invalid Json! (" + clientSocket.getInetAddress().getHostAddress() + ")\nCotent:\n" + content);
                        try {
                            server.getPacketManager().sendPacket(server, new PrintStream(clientSocket.getOutputStream()), new S7HandshakeResponse(null, false, "Invalid JSON"));
                        } catch (Exception ignore) { }

                        try {
                            inputScanner.close();
                            clientSocket.close();
                        } catch (IOException ignore) {
                        }
                    }

                } catch (Exception e) {
                    if (e instanceof IllegalStateException) {
                        Main.getInstance().getLogger().error("Couldn't handle connection! (" + clientSocket.getInetAddress().getHostAddress() + ", INPUT CLOSED)");
                    } else if (e instanceof JsonSyntaxException) {
                        Main.getInstance().getLogger().error("Couldn't handle connection! (" + clientSocket.getInetAddress().getHostAddress() + ", NO JSON)\nContent:\n" + (content != null ? content : "NULL"));
                    } else {
                        Main.getInstance().getLogger().error("Couldn't handle connection!(" + clientSocket.getInetAddress().getHostAddress() + ")\nContent:\n" + (content != null ? content : "NULL"), e);
                    }

                    try {
                        server.getPacketManager().sendPacket(server, new PrintStream(clientSocket.getOutputStream()), new S7HandshakeResponse(null, false, e.getMessage()));
                    } catch (Exception ignore) {
                    }

                    try {
                        inputScanner.close();
                        clientSocket.close();
                    } catch (IOException ignore) {
                    }
                }

                Main.getInstance().getLogger().warn("connection handler thread stopped");
            }
        });

        handlerThread.start();
    }


    public void stopHandler() {
        handlerThread.interrupt();
    }
}
